<?php

namespace Layout\Helper;

class Components extends \Lime\Helper {

    protected array $components = [];

    protected function initialize() {

        $components = new \ArrayObject($this->app['debug'] ? $this->cache(false) : $this->app->memory->get('layout.components', function() {
            return $this->cache();
        }));

        $this->app->trigger('layout.components.collect', [$components]);
        $this->components = $components->getArrayCopy();
    }

    public function components(): array {
        return $this->components;
    }

    public function cache(bool $persistent = true): array {

        $cache = $this->getDefaultComponents();

        $components = $this->app->dataStorage->find('layout/components', [
            'sort' => ['name' => 1]
        ])->toArray();

        foreach ($components as $component) {
            $cache[$component['name']] = $component['meta'];
        }

        if ($persistent) {
            $this->app->memory->set('layout.components', $cache);
        }

        return $cache;
    }

    public function process($array, array $context = []) {

        if (!\is_array($array)) {
            return $array;
        }

        // resolve shared component
        if (isset($array['shared'])) {

            $shared = $this->app->dataStorage->findOne('layout/components_shared', ['_id' => $array['shared']]);

            if ($shared) {
                $array = $this->app->helper('locales')->applyLocales(\array_replace_recursive($shared, $array), $context['locale'] ?? 'default');
                $array['label'] = $shared['name'];
                unset($array['_id'], $array['name']);
            }
        }

        foreach ($array as $k => $v) {

            if (\is_array($array[$k])) {
                $array[$k] = $this->process($array[$k], $context);
            }

            if ($k == 'component' && \is_string($v) && isset($this->components[$v])) {

                $this->app->trigger('layout.component.process', [$v, &$array, $context, $this->components[$v]]);
            }
        }

        return $array;
    }

    public function removeSharedComponents(array $array, string $sharedId): array {

        $isList = \array_is_list($array);

        if ($isList) {

            for ($i = \count($array) - 1; $i >= 0; $i--) {

                $item = $array[$i];

                if (\is_array($item) && isset($item['shared']) && $item['shared'] === $sharedId) {
                    \array_splice($array, $i, 1);
                } elseif (\is_array($item)) {
                    $array[$i] = $this->removeSharedComponents($item, $sharedId);
                }
            }

        } else {

            // Associative or non-sequential array => set matches to null
            foreach ($array as $key => $item) {
                if (\is_array($item) && isset($item['shared']) && $item['shared'] === $sharedId) {
                    $array[$key] = null;
                }elseif (\is_array($item)) {
                    $array[$key] = $this->removeSharedComponents($item, $sharedId);
                }
            }
        }

        return $array;
    }

    public function getDefaultComponents() {

        $isPagesAddonAvailable = $this->app->module('pages');

        return [

            'grid' => [

                'label' => 'Grid',
                'group' => 'Layout',
                'type'  => 'grid',
                'fields' => [
                    [
                        'name' => 'colWidth',
                        'type' => 'select',
                        'opts' => [
                            'options' => ['auto', 1, 2, 3, 4],
                        ],
                    ],
                ],
                'preview' => null,
                'children' => false,
                'opts' => [
                    // Custom per-column settings rendered in grid editor
                    // Example: [['name'=>'class','type'=>'text','label'=>'CSS Class']]
                    'colFields' => [],
                    // Optional column limits
                    'minCols' => null,
                    'maxCols' => null,
                ],
            ],

            'row' => [

                'label' => 'Row',
                'group' => 'Layout',
                'type'  => 'row',
                'fields' => [],
                'preview' => null,
                'children' => false,
                'opts' => [
                    // Define custom per-column fields in the row editor
                    // Keep empty by default to avoid duplicating the built-in width control
                    'colFields' => [],
                    // Optional column limits; default row max is 4 to preserve behavior
                    'minCols' => null,
                    'maxCols' => 4,
                ],
            ],

            'button' => [
                'label' => 'Button',
                'group' => 'Basic',
                'fields' => [
                    ['name' => 'url', 'type' => $isPagesAddonAvailable ? 'pageUrl' : 'text'],
                    ['name' => 'caption', 'type' => 'text'],
                    ['name' => 'target', 'type' => 'select', 'opts' => ['options' => ['_self', '_blank'], 'default' => '_self']],
                ],
                'preview' => null,
                'children' => false
            ],

            'divider' => [
                'label' => 'Divider',
                'group' => 'Basic',
                'fields' => [

                ],
                'preview' => null,
                'children' => false
            ],

                'heading' => [
                'label' => 'Heading',
                'group' => 'Text',
                'fields' => [
                    ['name' => 'text', 'type' => 'text', 'i18n' => true],
                    ['name' => 'level', 'type' => 'select', 'opts' => ['options' => [1,2,3,4,5,6]]],
                ],
                'preview' => '<div class="kiss-padding-small kiss-color-muted kiss-flex kiss-flex-middle" v-if="data.text"><strong class="kiss-margin-xsmall-end" v-if="data.level">H{{data.level}}</strong> {{ data.text }}</div>',
                'children' => false,
                'opts' => [
                    'previewComponent' => null
                ]
            ],

            'html' => [
                'label' => 'HTML',
                'group' => 'Misc',
                'fields' => [
                    ['name' => 'html', 'type' => 'code', 'i18n' => true, 'opts' => ['mode' => 'html']],
                ],
                'preview' => null,
                'children' => false,
                'opts' => [
                    'dialog.size' => 'xlarge',
                ]
            ],

            'image' => [
                'label' => 'Image',
                'group' => 'Media',
                'fields' => [
                    ['name' => 'asset', 'type' => 'asset', 'opts' => ['filter' => ['type' => 'image']]],
                    ['name' => 'url', 'label' => 'Link url', 'type' => $isPagesAddonAvailable ? 'pageUrl' : 'text', 'opts' => ['placeholder' => 'https://...']],
                ],
                'preview' => '<div class="kiss-bgcolor-transparentimage" v-if="data.asset"><img class="kiss-margin-auto" :src="App.route(`/assets/thumbnail/${data.asset._id}?m=bestFit&mime=auto&h=100&t=${data.asset._modified}`)"></div>',
                'children' => false
            ],

            'video' => [
                'label' => 'Video',
                'group' => 'Media',
                'fields' => [
                    ['name' => 'video', 'type' => 'asset', 'opts' => ['filter' => ['type' => 'video']]],
                    ['name' => 'poster', 'type' => 'asset', 'label' => 'Poster Image', 'opts' => ['filter' => ['type' => 'image']]],
                    ['name' => 'attr', 'label' => 'Attributes', 'info' => 'Video related attributes', 'type' => 'select', 'opts' => [
                        'multiple' => true,
                        'options' => ['autoplay', 'controls', 'loop', 'muted', 'playsinline']
                    ]],
                ],
                'preview' => null,
                'children' => false
            ],

            'markdown' => [
                'label' => 'Markdown',
                'group' => 'Text',
                'fields' => [
                    ['name' => 'markdown', 'type' => 'code', 'i18n' => true, 'opts' => ['mode' => 'markdown']],
                ],
                'preview' => '<div class="kiss-padding-small kiss-color-muted" v-if="data.markdown">{{ truncate(data.markdown, 50) }}</div>',
                'children' => false,
                'opts' => [
                    'previewComponent' => null
                ]
            ],

                'link' => [
                'label' => 'Link',
                'group' => 'Basic',
                'fields' => [
                    ['name' => 'url', 'type' => $isPagesAddonAvailable ? 'pageUrl' : 'text'],
                    ['name' => 'caption', 'type' => 'text'],
                    ['name' => 'target', 'type' => 'select', 'opts' => ['options' => ['_self', '_blank'], 'default' => '_self']],
                ],
                'preview' => null,
                'children' => false
            ],

                'richtext' => [
                'label' => 'Richtext',
                'group' => 'Text',
                'fields' => [
                    ['name' => 'html', 'type' => 'wysiwyg', 'i18n' => true],
                ],
                'preview' => '<div class="kiss-padding-small kiss-color-muted kiss-overflow-y-auto" style="max-height:250px" v-if="data.html" v-html="stripTags(data.html, \'<br><p><strong><i><div><em><hr><code><h1><h2><h3><h4><h5><h6>\')"></div>',
                'children' => false,
                'opts' => [
                    'dialog.size' => 'xlarge',
                    'previewComponent' => null,
                ]
            ],

                'section' => [
                'label' => 'Section',
                'group' => 'Layout',
                'fields' => [],
                'preview' => null,
                'children' => true
            ],

                'spacer' => [
                'label' => 'Spacer',
                'group' => 'Layout',
                'fields' => [
                    ['name' => 'size', 'type' => 'text'],
                ],
                'preview' => '<div class="kiss-padding-small kiss-color-muted kiss-flex kiss-flex-middle" v-if="data.size"><icon class="kiss-margin-xsmall-end">unfold_more</icon>{{ data.size }}</div>',
                'children' => false,
                'opts' => [
                    'previewComponent' => null
                ]
            ]
        ];
    }
}
