<?php

namespace Layout\Controller;

use ArrayObject;

class Settings extends Controller {

    protected function before() {

        // Use the same permission key as registered in admin.php
        if (!$this->isAllowed('layout/components/manage')) {
            return $this->stop(401);
        }
    }

    public function index() {
        return $this->render('layout:views/components/index.php');
    }

    public function component($id = null) {

        $component = $this->getDefaultComponentSettings();

        if ($id) {

            $component = $this->app->dataStorage->findOne('layout/components', ['_id' => $id]);

            if (!$component) {
                return false;
            }

            $this->checkAndLockResource($id);
        }

        return $this->render('layout:views/components/component.php', \compact('component'));
    }

    public function clone($id = null) {

        if (!$id) {
            return false;
        }

        $component = $this->app->dataStorage->findOne('layout/components', ['_id' => $id]);

        if (!$component) {
            return false;
        }

        unset($component['_id'], $component['_created'], $component['_modified']);
        $component['name'] = '';

        if (isset($component['meta']['label']) && $component['meta']['label']) {
            $component['meta']['label'] .= ' (Clone)';
        }

        return $this->render('layout:views/components/component.php', \compact('component'));
    }

    public function remove() {

        $this->hasValidCsrfToken(true);

        $component = $this->param('component');

        if (!$component || !isset($component['_id'], $component['name'])) {
            return $this->stop(['error' => 'component is missing'], 412);
        }

        $this->app->dataStorage->remove('layout/components', ['_id' => $component['_id']]);

        $this->app->trigger('layout.components.remove', [$component]);

        $this->cache();

        return ['success' => true];
    }

    public function save() {

        $this->hasValidCsrfToken(true);

        $component = $this->param('component');

        if (!$component) {
            return $this->stop(['error' => 'Component data is missing'], 412);
        }

        $component['_modified'] = \time();
        $isUpdate = isset($component['_id']);

        if (!$isUpdate) {
            $component['_created'] = $component['_modified'];
        }

        if (!isset($component['name']) || !\trim($component['name'])) {
            return $this->stop(['error' => 'name required'], 412);
        }

        foreach (['name'] as $key) {
            $component[$key] = \strip_tags(\trim($component[$key]));
        }

        // unique check

        $_component = $this->app->dataStorage->findOne('layout/components', ['name' => $component['name']]);

        if ($_component && (!isset($component['_id']) || $component['_id'] != $_component['_id'])) {
            return $this->app->stop(['error' => 'Component name is already used!'], 412);
        }

        $this->app->trigger('layout.components.save', [&$component, $isUpdate]);
        $this->app->dataStorage->save('layout/components', $component);

        $component = $this->app->dataStorage->findOne('layout/components', ['_id' => $component['_id']]);

        $component['meta'] = new ArrayObject(\is_array($component['meta']) ? $component['meta'] : []);
        $component['meta']['opts'] = new ArrayObject(\is_array($component['meta']['opts']) ? $component['meta']['opts'] : []);

        $this->cache();

        return $component;
    }

    public function load() {

        $this->helper('session')->close();
        $this->hasValidCsrfToken(true);

        $components = $this->app->dataStorage->find('layout/components', [
            'sort' => ['name' => 1]
        ])->toArray();

        return $components;
    }

    protected function getDefaultComponentSettings(): array {

        return [
            'name' => '',
            'meta' => [
                'icon'     => null,
                'label'    => '',
                'group'    => '',
                'fields'   => [],
                'preview'  => null,
                'children' => false,
                'opts'     => []
            ]
        ];
    }

    protected function cache() {
        $this->helper('layoutComponents')->cache();
    }
}
