# Webhooks

[![Commercial](https://img.shields.io/badge/License-Commercial-red.svg)](LICENSE.md)
[![Cockpit CMS](https://img.shields.io/badge/Cockpit%20CMS-Addon-blue.svg)](https://getcockpit.com)
[![Webhooks](https://img.shields.io/badge/Feature-HTTP%20Integration-green.svg)](#features)

> **Event-driven HTTP webhook system for Cockpit CMS**

Webhooks enables seamless integration between Cockpit CMS and external services through HTTP callbacks. Automatically trigger web requests when events occur in your CMS, enabling real-time synchronization, notifications, and workflow automation.

## ✨ Features

### 🔗 **HTTP Integration**
- **Multiple HTTP Methods**: Support for GET, POST, PUT, DELETE, PATCH
- **Custom Headers**: Configure authentication and custom request headers
- **Flexible Payloads**: Send event data or custom JSON payloads
- **Environment Variables**: Secure configuration with DotEnv support

### ⚡ **Event-Driven Architecture**
- **Any Cockpit Event**: Hook into any system event (content.save, user.login, etc.)
- **Multiple Events**: Single webhook can respond to multiple events
- **Event Filtering**: Conditional webhook execution
- **Real-time Triggers**: Immediate or background execution

### 🚀 **Performance & Reliability**
- **Async Execution**: Background processing via Worker or Async helpers
- **Timeout Control**: Configurable request timeouts (default 5 seconds)
- **Error Handling**: Comprehensive error logging and response tracking
- **Debug Support**: Generate curl commands for testing

### 🔧 **Advanced Features**
- **SSL Support**: Automatic HTTPS handling and SSL verification
- **Response Tracking**: Full response headers, status codes, and timing
- **Environment Resolution**: Dynamic URL and header value resolution
- **Conditional Execution**: Enable/disable webhooks individually

## 🚀 Quick Start

### 1. Basic Webhook Configuration

Create a webhook through the admin interface or programmatically:

```php
// Basic webhook configuration
$webhook = [
    'name' => 'content-sync',
    'enabled' => true,
    'url' => 'https://api.example.com/webhook',
    'method' => 'POST',
    'events' => ['content.save.after', 'content.remove.after'],
    'headers' => [
        ['key' => 'Authorization', 'val' => 'Bearer ${API_TOKEN}'],
        ['key' => 'Content-Type', 'val' => 'application/json']
    ],
    'body' => [
        'payload' => true,
        'custom' => false,
        'contents' => []
    ]
];

$app->dataStorage->save('webhooks', $webhook);
```

### 2. Environment Variables

Store sensitive data in environment variables:

```bash
# .env file
API_TOKEN=your-secret-api-token
WEBHOOK_URL=https://api.example.com/webhook
WEBHOOK_SECRET=webhook-signing-secret
```

Use in webhook configuration:
```php
'url' => '${WEBHOOK_URL}',
'headers' => [
    ['key' => 'Authorization', 'val' => 'Bearer ${API_TOKEN}'],
    ['key' => 'X-Webhook-Secret', 'val' => '${WEBHOOK_SECRET}']
]
```

### 3. Event Listening

Webhooks automatically listen to configured events:

```php
// This will trigger all webhooks listening to 'content.save.after'
$app->trigger('content.save.after', [$content, $isUpdate]);

// Custom event triggering
$app->trigger('custom.event', ['data' => $customData]);
```

## 📋 Webhook Configuration

### Basic Properties

```php
[
    'name' => 'webhook-name',           // Unique identifier
    'enabled' => true,                  // Enable/disable webhook
    'url' => 'https://api.site.com',   // Target URL (supports env vars)
    'method' => 'POST',                 // HTTP method (GET, POST, PUT, DELETE, PATCH)
    'events' => [                       // Array of events to listen to
        'content.save.after',
        'user.login.after'
    ]
]
```

### Headers Configuration

```php
'headers' => [
    ['key' => 'Authorization', 'val' => 'Bearer ${API_TOKEN}'],
    ['key' => 'Content-Type', 'val' => 'application/json'],
    ['key' => 'X-Webhook-Source', 'val' => 'cockpit-cms'],
    ['key' => 'X-Timestamp', 'val' => '${TIMESTAMP}']
]
```

### Payload Configuration

**Send Event Data** (default):
```php
'body' => [
    'payload' => true,      // Send event payload
    'custom' => false,      // Not using custom data
    'contents' => []        // Empty (event data will be sent)
]
```

**Custom Payload**:
```php
'body' => [
    'payload' => true,      // Enable payload
    'custom' => true,       // Use custom data
    'contents' => [         // Custom JSON data
        'source' => 'cockpit',
        'environment' => '${APP_ENV}',
        'timestamp' => time()
    ]
]
```

**No Payload** (GET requests or notification-only):
```php
'body' => [
    'payload' => false,     // No payload
    'custom' => false,
    'contents' => []
]
```

## 🎯 Common Use Cases

### 1. Content Synchronization

Sync content changes to external systems:

```php
$webhook = [
    'name' => 'content-sync',
    'enabled' => true,
    'url' => 'https://cdn.example.com/api/invalidate',
    'method' => 'POST',
    'events' => [
        'content.save.after',
        'content.remove.after',
        'assets.save.after'
    ],
    'headers' => [
        ['key' => 'Authorization', 'val' => 'Bearer ${CDN_API_KEY}'],
        ['key' => 'Content-Type', 'val' => 'application/json']
    ],
    'body' => [
        'payload' => true,
        'custom' => false
    ]
];
```

### 2. User Notifications

Send notifications on user events:

```php
$webhook = [
    'name' => 'slack-notifications',
    'enabled' => true,
    'url' => '${SLACK_WEBHOOK_URL}',
    'method' => 'POST',
    'events' => ['user.login.after', 'user.logout.after'],
    'headers' => [
        ['key' => 'Content-Type', 'val' => 'application/json']
    ],
    'body' => [
        'payload' => true,
        'custom' => true,
        'contents' => [
            'channel' => '#admin-alerts',
            'username' => 'Cockpit CMS',
            'icon_emoji' => ':gear:'
        ]
    ]
];
```

### 3. Cache Invalidation

Trigger cache clearing on content updates:

```php
$webhook = [
    'name' => 'cache-invalidation',
    'enabled' => true,
    'url' => 'https://api.cloudflare.com/client/v4/zones/${ZONE_ID}/purge_cache',
    'method' => 'POST',
    'events' => ['content.save.after'],
    'headers' => [
        ['key' => 'Authorization', 'val' => 'Bearer ${CLOUDFLARE_TOKEN}'],
        ['key' => 'Content-Type', 'val' => 'application/json']
    ],
    'body' => [
        'payload' => true,
        'custom' => true,
        'contents' => [
            'purge_everything' => true
        ]
    ]
];
```

### 4. Analytics Tracking

Send usage data to analytics platforms:

```php
$webhook = [
    'name' => 'analytics-tracking',
    'enabled' => true,
    'url' => 'https://api.analytics.com/events',
    'method' => 'POST',
    'events' => ['app.request.after'],
    'headers' => [
        ['key' => 'X-API-Key', 'val' => '${ANALYTICS_KEY}'],
        ['key' => 'Content-Type', 'val' => 'application/json']
    ],
    'body' => [
        'payload' => true,
        'custom' => true,
        'contents' => [
            'event_type' => 'page_view',
            'source' => 'cockpit-cms'
        ]
    ]
];
```

## ⚙️ Advanced Configuration

### Worker Integration

Enable background processing for better performance:

```php
// In config
'webhooks' => [
    'worker' => true
]
```

When enabled, webhooks execute via the Worker addon instead of blocking the main request.

### Custom Event Filtering

Create conditional webhooks:

```php
// Custom event handler with filtering
$this->on('content.save.after', function($content, $isUpdate) {
    
    // Only trigger for published content
    if ($content['_state'] !== 1) {
        return;
    }
    
    // Only trigger for specific collections
    if (!in_array($content['_collection'], ['blog', 'news'])) {
        return;
    }
    
    // Trigger custom event
    $this->trigger('content.published', [$content]);
});
```

### Environment Variable Resolution

Webhooks support dynamic environment variable resolution:

```php
// These will be resolved at runtime:
'url' => 'https://${API_HOST}/webhook/${API_VERSION}',
'headers' => [
    ['key' => 'Authorization', 'val' => 'Bearer ${API_TOKEN}'],
    ['key' => 'X-Environment', 'val' => '${APP_ENV}'],
    ['key' => 'X-Timestamp', 'val' => '${CURRENT_TIME}']
]
```

### SSL Configuration

Control SSL verification for development/testing:

```php
// HTTPS URLs automatically enable SSL verification
'url' => 'https://api.example.com/webhook'  // SSL verified

// HTTP URLs disable SSL verification
'url' => 'http://localhost:3000/webhook'    // SSL not verified
```

## 🔧 Programmatic Usage

### Execute Webhooks Manually

```php
// Get webhook
$webhook = $app->dataStorage->findOne('webhooks', ['name' => 'content-sync']);

// Execute with custom payload
$result = $app->module('webhooks')->execute($webhook, [
    'action' => 'content_updated',
    'timestamp' => time(),
    'data' => $customData
]);

// Check result
if ($result['success']) {
    echo "Webhook executed successfully";
    echo "Status: " . $result['status_code'];
    echo "Duration: " . $result['duration_ms'] . "ms";
} else {
    echo "Webhook failed: " . $result['error']['message'];
}
```

### Generate Debug Commands

```php
// Generate curl command for testing
$result = $app->module('webhooks')->execute($webhook, $payload, [
    'curl_command' => true
]);

echo $result['curl_command'];
// Output: curl -X POST -H "Authorization: Bearer token" -d '{"data":"value"}' https://api.example.com/webhook
```

### Webhook Response Handling

```php
$result = $app->module('webhooks')->execute($webhook, $payload);

// Access response details
echo "Status Code: " . $result['status_code'];
echo "Success: " . ($result['success'] ? 'Yes' : 'No');
echo "Duration: " . $result['duration_ms'] . "ms";
echo "Response Body: " . $result['content'];

// Access response headers
foreach ($result['headers'] as $name => $value) {
    echo $name . ": " . $value;
}

// Handle errors
if (!$result['success'] && isset($result['error'])) {
    echo "Error Code: " . $result['error']['code'];
    echo "Error Message: " . $result['error']['message'];
}
```

## 📊 Events Reference

### System Events

Common Cockpit events you can hook into:

```php
// Application lifecycle
'app.init'                  // Application initialization
'app.request.before'        // Before request processing
'app.request.after'         // After request processing

// Authentication
'user.login.before'         // Before user login
'user.login.after'          // After successful login
'user.logout.after'         // After user logout

// Content operations
'content.save.before'       // Before content save
'content.save.after'        // After content save
'content.remove.before'     // Before content deletion
'content.remove.after'      // After content deletion

// Asset operations
'assets.save.after'         // After asset upload/update
'assets.remove.after'       // After asset deletion

// Collection-specific events
'collections.{name}.save.before'    // Before collection item save
'collections.{name}.save.after'     // After collection item save
'collections.{name}.remove.after'   // After collection item deletion

// Singleton-specific events
'singletons.{name}.save.after'      // After singleton update

// Form submissions (Inbox addon)
'forms.submit.after'        // After form submission

// User management
'accounts.save.after'       // After user account save
'accounts.remove.after'     // After user account deletion
```

### Custom Events

Create and trigger custom events:

```php
// Define custom event
$this->on('custom.order.completed', function($order, $customer) {
    // Custom logic here
});

// Trigger custom event
$this->trigger('custom.order.completed', [$orderData, $customerData]);
```


## 🔒 Security Considerations

### Environment Variables

Store sensitive data in environment variables:

```bash
# .env
API_TOKEN=your-secret-token
WEBHOOK_SECRET=signing-secret
SLACK_WEBHOOK_URL=https://hooks.slack.com/services/...
```

### Webhook Verification

Implement webhook signature verification:

```php
// In your webhook endpoint
$signature = $_SERVER['HTTP_X_WEBHOOK_SIGNATURE'] ?? '';
$payload = file_get_contents('php://input');
$secret = getenv('WEBHOOK_SECRET');

$expectedSignature = hash_hmac('sha256', $payload, $secret);

if (!hash_equals($signature, $expectedSignature)) {
    http_response_code(401);
    exit('Invalid signature');
}
```

### URL Validation

Webhooks validate URLs before execution:

```php
// Valid URLs
'https://api.example.com/webhook'    // ✅ HTTPS
'http://localhost:3000/dev'          // ✅ Local development

// Invalid URLs  
'javascript:alert(1)'               // ❌ Invalid protocol
'ftp://example.com'                 // ❌ Unsupported protocol
'not-a-url'                         // ❌ Invalid format
```

## 🐛 Troubleshooting

### Common Issues

**❌ Webhook not triggering**
- Check webhook is enabled (`enabled: true`)
- Verify event names match exactly
- Ensure events are being triggered in your code
- Check webhook filtering conditions

**❌ Connection timeouts**
- Default timeout is 5 seconds
- Check target server response time
- Verify network connectivity
- Consider using async execution

**❌ SSL certificate errors**
- Ensure HTTPS URLs have valid certificates
- For development, use HTTP URLs
- Check SSL certificate chain

**❌ Authentication failures**
- Verify API tokens and secrets
- Check header configuration
- Ensure environment variables are loaded
- Test authentication manually with curl


## 📄 License

This is a commercial addon for Cockpit CMS. See [LICENSE.md](LICENSE.md) for full terms.

## 🙏 Credits

Webhooks is developed by [Agentejo](https://agentejo.com) as part of the Cockpit CMS Pro ecosystem.

---

**Ready to integrate?** Set up your webhooks and connect Cockpit CMS to your external services for real-time automation!