export default {

    props: {
        forms: {
            type: Array,
            default: () => []
        },
        limit: {
            type: Number,
            default: 5
        },
        open: {
            type: Number,
            default: 1
        }
    },

    data() {
        return {
            loading: true,
        }
    },

    computed: {
        formsWithSubmissions() {

            let forms = this.forms
                .filter(form => form.submissions && form.submissions.length > 0)
                .sort((a, b) => {
                    const timeA = a.submissions[0]._created || 0;
                    const timeB = b.submissions[0]._created || 0;
                    return timeB - timeA;
                });

            forms.forEach((f, i) => {
                if (f._visibleSubmissions === undefined) {
                    f._visibleSubmissions = i < this.open;
                }
            });

            return forms;
        }
    },

    watch: {
        forms: {
            handler() {
                this.load();
            },
            deep: false
        }
    },

    mounted() {
        this.load();
    },

    methods: {
        async load() {

            let promises = [];

            this.loading = true;

            (this.forms || []).forEach(form => {

                const p = new Promise((resolve, reject) => {

                    let options = {
                        sort: { _created: -1 },
                        limit: this.limit,
                    };

                    this.$request(`/inbox/submissions/load/${form._id}`, { options }).then(rsp => {
                        form.submissions = rsp.items;
                        resolve();
                    }).catch(() => resolve());
                });

                promises.push(p);
            });

            Promise.all(promises).then(() => {
                this.loading = false;
            });
        },

        preview(submission) {
            return JSON.stringify(Object.values(submission.data || {})).replace(/('null'|\[|\]|\{|\}|"|\\|')/g, '').replace(/,/g, ', ').substr(0, 50).trim();
        },

        showSubmission(submission) {
            VueView.ui.offcanvas('inbox:assets/dialogs/submission.js', { submission });
        }
    },

    template: /*html*/`
        <div>

            <div class="kiss-padding-large" v-if="loading"><app-loader size="small"></app-loader></div>

            <div class="animated fadeIn kiss-padding-large kiss-flex kiss-flex-middle kiss-flex-center kiss-align-center kiss-color-muted kiss-margin" v-if="!loading && !formsWithSubmissions.length">
                <div>
                    <kiss-svg :src="$baseUrl('inbox:icon.svg')" width="30" height="30"></kiss-svg>
                    <p class="kiss-size-small kiss-margin-small-top">{{ t('No submissions') }}</p>
                </div>
            </div>

            <div class="animated fadeIn kiss-margin-small" v-if="!loading">

                <kiss-card class="kiss-padding kiss-margin-small" theme="bordered" hover="contrast shadowed bordered-primary" v-for="form in formsWithSubmissions">

                    <div class="kiss-flex kiss-flex-middle kiss-margin kiss-visible-toggle" gap="small">
                        <div>
                            <kiss-svg :src="$baseUrl(form.icon ? form.icon : 'inbox:icon.svg')" width="25" height="25" :style="{color:form.color ? form.color : 'var(--kiss-color-muted)'}"><canvas width="25" height="25"></canvas></kiss-svg>
                        </div>
                        <a class="kiss-text-bold kiss-text-capitalize kiss-link-muted" :href="$routeUrl('/inbox/submissions/form/'+form._id)">{{ form.name || form.label }}</a>
                        <div class="kiss-flex-1"></div>
                        <a class="kiss-link-muted kiss-flex kiss-flex-middle" gap="small" @click="form._visibleSubmissions = !form._visibleSubmissions">
                            <span class="kiss-badge">{{ form.submissions.length }}</span>
                            <icon>unfold_more</icon>
                        </a>
                    </div>

                    <div class="kiss-margin-small" v-if="form._visibleSubmissions">
                        <div class="kiss-flex kiss-margin-small kiss-size-small" gap="small" v-for="submission in form.submissions">
                            <icon class="kiss-color-muted">mail</icon>
                            <div>
                                <a class="kiss-link-muted kiss-flex-1" @click="showSubmission(submission)">{{ preview(submission) }}</a>
                                <div class="kiss-text-monospace kiss-color-muted kiss-size-xsmall"><app-datetime type="relative" :datetime="submission._created" /></div>
                            </div>
                        </div>
                    </div>

                </kiss-card>

            </div>
        </div>
    `
}
