export default {

    name: 'grid-layout',

    data() {

        let val = this.modelValue || {};

        if (!val.columns || !Array.isArray(val.columns)) {
            val.columns = [];
        }

        return {
            val,
            active: false,
            showPreview: this.preview,
            globalComponents: null
        }
    },

    props: {
        modelValue: {
            type: Object,
            default: {}
        },
        group: {
            type: String,
            default: null
        },
        level: {
            type: Number,
            default: 0
        },
        preview: {
            type: Boolean,
            default: false
        },
        allowedComponents: {
            type: Array,
            default: []
        },
    },

    watch: {
        val: {
            handler() { this.update() },
            deep: true
        },
        modelValue: {
            handler(val) {
                this.val = this.modelValue || {};
                if (!this.val.columns) {
                    this.val.columns = [];
                }
                this.ensureMinColumns();
                this.update();
            },
            deep: true
        },
        preview(val) {
            this.showPreview = val;
        }
    },

    computed: {
        cols() {
            return (this.val.data && this.val.data.colWidth) || 'auto';
        },
        meta() {
            return this.globalComponents && this.val && this.val.component
                ? this.globalComponents[this.val.component] || null
                : null;
        },
        colFields() {
            return (this.meta && this.meta.opts && Array.isArray(this.meta.opts.colFields))
                ? this.meta.opts.colFields
                : [];
        },
        minCols() {
            const v = this.meta && this.meta.opts ? this.meta.opts.minCols : null;
            return Number.isInteger(v) && v > 0 ? v : 0;
        },
        maxCols() {
            const v = this.meta && this.meta.opts ? this.meta.opts.maxCols : null;
            return Number.isInteger(v) && v > 0 ? v : null;
        }
    },

    methods: {

        addColumn(idx = 0) {

            let column = {
                data: {},
                meta: {},
                components: []
            };

            if (this.maxCols && (this.val.columns.length + 1) > this.maxCols) {
                App.ui.notify(this.t ? this.t('Max {max} columns allowed', {max: this.maxCols}) : `Max ${this.maxCols} columns allowed`, 'error');
                return;
            }

            if (!this.val.columns.length) {
                this.val.columns.push(column);
            } else {
                this.val.columns.splice(idx + 1, 0, column);
            }

        },

        update() {
            this.$emit('update:modelValue', this.val)
        },

        ensureMinColumns() {
            if (!Array.isArray(this.val.columns)) this.val.columns = [];
            while (this.val.columns.length < this.minCols) {
                this.val.columns.push({ data: {}, meta: {}, components: [] });
            }
        },

        editColumn(column) {
            VueView.ui.modal('layout:assets/vue-components/layout/column-edit.js', {
                data: column.data || {},
                meta: column.meta || {},
                fields: this.colFields,
                title: this.t ? this.t('Column settings') : 'Column settings'
            }, {
                save: payload => {
                    column.data = payload?.data || {};
                    column.meta = payload?.meta || {};
                }
            }, { size: 'large' });
        }
    },

    mounted() {
        // Load component meta to know about custom per-column fields
        App.utils.getlayoutComponents().then(components => {
            this.globalComponents = components;
            this.ensureMinColumns();
        });
    },

    template: /*html*/`
        <kiss-card>

            <kiss-grid :cols="cols" gap="small">
                <div class="kiss-flex-1 app-border-radius kiss-bgcolor-contrast kiss-padding-xsmall" v-for="(column, idx) in val.columns">
                    <kiss-card>
                        <div class="kiss-margin-xsmall kiss-flex kiss-flex-middle">
                            <a class="kiss-size-xsmall kiss-flex kiss-link-muted kiss-margin-small-start kiss-hidden-hover" @click="editColumn(column)" :title="t('Column settings')"><icon>tune</icon></a>
                            <div class="kiss-flex-1"></div>
                            <a class="kiss-size-xsmall kiss-flex kiss-color-danger kiss-margin-small-start kiss-hidden-hover" v-if="val.columns.length > minCols" @click="val.columns.splice(val.columns.indexOf(column), 1)" :title="t('Remove column')"><icon>remove_circle</icon></a>
                            <a class="kiss-size-xsmall kiss-flex kiss-link-muted kiss-margin-xsmall-start kiss-hidden-hover" @click="addColumn(idx)" :title="t('Add column')"><icon>add_circle</icon></a>
                        </div>
                        <field-layout v-model="column.components" :group="group" :allowedComponents="allowedComponents" :level="level + 1" :preview="showPreview"></field-layout>
                    </kiss-card>
                </div>
            </kiss-grid>

            <div class="kiss-padding kiss-align-center kiss-color-muted" v-if="!val.columns.length">
                <button type="button" class="kiss-button kiss-button-small kiss-margin-small" @click="addColumn(0)">{{ t('Add column') }}</button>
            </div>
        </kiss-card>
    `
}
