export default {
    props: {
        limit: {
            type: Number,
            default: 4
        },

        locale: {
            type: String,
            default: null
        }
    },

    data() {
        return {
            pages: [],
            loading: true,
            mode: 'all',
            state: null,
        };
    },

    mounted() {
        this.load();
    },

    computed: {
        stateText() {
            return this.state === null ? this.t('All') : this.state === 1 ? this.t('Published') : this.t('Unpublished');
        }
    },

    watch: {
        mode() {
            this.load();
        },
        state() {
            this.load();
        },
        locale() {
            this.load();
        }
    },

    methods: {
        async load() {

            this.loading = true;

            let options = {
                limit: this.limit,
                sort: {
                    _modified: -1
                }
            };

            if (this.mode === 'byme') {
                options.filter = {
                    _mby: App.user._id
                };
            }

            if (this.locale) {
                options.locale = this.locale;
            }

            if (this.state !== null) {
                options.filter = options.filter || {};
                options.filter._state = this.state;
            }

            this.$request('/pages/utils/find', { options }).then(res => {
                this.pages = res.items;
            }).catch(res => {
                App.ui.notify(res.error || 'Loading failed!', 'error');
            }).finally(() => {
                this.loading = false;
            });
        }
    },

    template: /*html*/`
        <div>
            <div class="kiss-flex kiss-flex-middle kiss-margin-small" gap>

                <ul class="kiss-tabs-nav kiss-flex-1" :class="{'kiss-disabled':loading}">
                    <li :active="mode == 'all' ? 'true':'false'"><a class="kiss-tabs-nav-link" @click="mode='all'">{{ t('All pages') }}</a></li>
                    <li :active="mode == 'all' ? 'false':'true'"><a class="kiss-tabs-nav-link"  @click="mode='byme'">{{ t('By me') }}</a></li>
                </ul>

                <div class="kiss-overlay-input">
                    <span class="kiss-badge" :class="{'kiss-bgcolor-success': state === 1, 'kiss-bgcolor-danger': state === 0}">
                        {{ stateText }}
                        <icon size="large">arrow_drop_down</icon>
                    </span>
                    <select v-model="state">
                        <option :value="null">{{ t('All') }}</option>
                        <hr>
                        <option :value="1">{{ t('Published') }}</option>
                        <option :value="0">{{ t('Unpublished') }}</option>
                    </select>
                </div>
            </div>

            <div class="kiss-padding-large" v-if="loading"><app-loader size="small"></app-loader></div>

            <div class="kiss-margin-small animated fadeIn" v-if="!loading">
                <kiss-card class="kiss-flex kiss-padding-small kiss-margin-xsmall kiss-visible-toggle" gap="small" theme="bordered" hover="contrast bordered-primary" v-for="page in pages" :key="page._id">
                    <icon class="kiss-size-small" :class="{'kiss-color-danger': !page._state, 'kiss-color-success': page._state === 1}">circle</icon>
                    <a class="kiss-link-muted kiss-size-small kiss-flex-1" :href="$routeUrl('/pages/page/'+page._id)">
                        <div class="kiss-text-bolder kiss-text-truncate">{{ page.title }}</div>
                        <div class="kiss-color-muted kiss-text-monospace kiss-size-xsmall kiss-flex kiss-flex-middle" gap="small">
                            <div class="kiss-flex-1"><app-datetime type="relative" :datetime="page._modified" /></div>
                            <div class="kiss-text-caption kiss-invisible-hover">{{ page.type }}</div>
                        </div>
                    </a>
                </kiss-card>
            </div>
        </div>
    `
}
