export default {

    name: 'row-layout',

    data() {

        let val = this.modelValue || {};

        if (!val.columns || !Array.isArray(val.columns)) {
            val.columns = [];
        }

        return {
            val,
            showPreview: this.preview,
            globalComponents: null
        }
    },

    props: {
        modelValue: {
            type: Object,
            default: {}
        },
        group: {
            type: String,
            default: null
        },
        level: {
            type: Number,
            default: 0
        },
        preview: {
            type: Boolean,
            default: false
        },
        allowedComponents: {
            type: Array,
            default: []
        },
    },

    watch: {
        val: {
            handler() { this.update() },
            deep: true
        },
        modelValue: {
            handler(val) {
                this.val = this.modelValue || {};
                if (!this.val.columns) {
                    this.val.columns = [];
                }
                this.ensureMinColumns();
                this.update();
            },
            deep: true
        },
        preview(val) {
            this.showPreview = val;
        }
    },

    computed: {
        meta() {
            return this.globalComponents && this.val && this.val.component
                ? this.globalComponents[this.val.component] || null
                : null;
        },
        colFields() {
            return (this.meta && this.meta.opts && Array.isArray(this.meta.opts.colFields))
                ? this.meta.opts.colFields
                : [];
        },
        hasWidthField() {
            return this.colFields.some(f => f && f.name === 'width');
        },
        minCols() {
            const v = this.meta && this.meta.opts ? this.meta.opts.minCols : null;
            return Number.isInteger(v) && v > 0 ? v : 0;
        },
        maxCols() {
            const v = this.meta && this.meta.opts ? this.meta.opts.maxCols : null;
            return Number.isInteger(v) && v > 0 ? v : 4;
        }
    },

    methods: {

        addColumn(idx = 0) {

            if (this.maxCols && (this.val.columns.length + 1) > this.maxCols) {
                App.ui.notify(this.t ? this.t('Max {max} columns allowed', {max: this.maxCols}) : `Max ${this.maxCols} columns allowed`, 'error');
                return;
            }

            let column = {
                data: {
                    width: null
                },
                meta: {},
                components: []
            };

            if (!this.val.columns.length) {
                this.val.columns.push(column);
            } else {
                this.val.columns.splice(idx + 1, 0, column);
            }

        },

        update() {
            this.$emit('update:modelValue', this.val)
        },

        ensureMinColumns() {
            if (!Array.isArray(this.val.columns)) this.val.columns = [];
            while (this.val.columns.length < this.minCols) {
                this.val.columns.push({ data: { width: null }, meta: {}, components: [] });
            }
        },

        editColumn(column) {

            VueView.ui.modal('layout:assets/vue-components/layout/column-edit.js', {
                data: column.data || {},
                fields: this.colFields,
                title: this.t ? this.t('Column settings') : 'Column settings'
            }, {
                save: data => {
                    column.data = data || {};
                }
            }, { size: 'large' });
        },
    },

    mounted() {
        // Load component meta to know about custom per-column fields
        App.utils.getlayoutComponents().then(components => {
            this.globalComponents = components;
            this.ensureMinColumns();
        });
    },

    template: /*html*/`
        <kiss-card>

            <kiss-row gap="small" match="true">
                <div class="app-border-radius kiss-bgcolor-contrast kiss-padding-xsmall" :class="column.data.width ? 'kiss-width-'+column.data.width: 'kiss-flex-1'" v-for="(column, idx) in val.columns">
                    <kiss-card>
                        <div class="kiss-flex kiss-flex-middle kiss-margin-xsmall">
                            <div class="kiss-overlay-input" v-if="!hasWidthField">
                                <span class="kiss-text-caption kiss-flex kiss-flex-middle">{{ column.data.width || 'auto' }}</span>
                                <select v-model="column.data.width">
                                    <option :value="null">Auto</option>
                                    <option value="1-2">1-2</option>
                                    <option value="1-3">1-3</option>
                                    <option value="1-4">1-4</option>
                                    <option value="2-3">2-3</option>
                                    <option value="3-4">3-4</option>
                                </select>
                            </div>
                            <div class="kiss-flex-1"></div>
                            <a class="kiss-size-xsmall kiss-flex kiss-link-muted kiss-margin-small-start kiss-hidden-hover" @click="editColumn(column)" :title="t('Column settings')"><icon>tune</icon></a>
                            <a class="kiss-size-xsmall kiss-flex kiss-color-danger kiss-margin-small-start kiss-hidden-hover" v-if="val.columns.length > minCols" @click="val.columns.splice(val.columns.indexOf(column), 1)" :title="t('Remove column')"><icon>remove_circle</icon></a>
                            <a class="kiss-size-xsmall kiss-flex kiss-link-muted kiss-margin-xsmall-start kiss-hidden-hover" @click="addColumn(idx)" :title="t('Add column')"><icon>add_circle</icon></a>
                        </div>
                        <div class="kiss-margin-small" v-if="false && colFields.length">
                            <!-- deprecated inline column fields-renderer replaced by modal editor -->
                        </div>
                        <field-layout v-model="column.components" :group="group" :allowedComponents="allowedComponents" :level="level + 1" :preview="showPreview"></field-layout>
                    </kiss-card>
                </div>
            </kiss-row>

            <div class="kiss-padding kiss-align-center kiss-color-muted" v-if="!val.columns.length">
                <button type="button" class="kiss-button kiss-button-small kiss-margin-small" @click="addColumn(0)">{{ t('Add column') }}</button>
            </div>
        </kiss-card>
    `
}
